/*
 * Mirror Test Data Reduction program - 
 * Version 2.3
 */
 /***************
 */
#define DO_GRAPHICS
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <dos.h>

#ifdef DO_GRAPHICS
#include <graphics.h>
#endif

#include <conio.h>

#define NRMAX 20    /* maximum number of sets of readings */
#define NZ 10	/* maximum number of zones */
#define MAX(A,B) ((A) > (B) ? (A) : (B))
#define MIN(A,B) ((A) < (B) ? (A) : (B))
#define RL 50
#define NSIM 8000
#define NEUTRAL 0
#define ABOVE 1
#define BELOW 2
#define TXT "mirror.txt"

#define QUIT 'Q'
#define APETURE_FILE 'A'
#define METHOD 'B'
#define EF_METHOD 'E'
#define CALC_TABLE  'C'
#define MIRROR_DATA 'D'
#define MIRROR_FILE 'F'
#define KEYBOARD 	'K'
#define MONTY_CARLO 'M'
#define GRAPHS      'G'
#define SAVE_SPECS  'S'
#define TEX_METHOD 'T'
#define ZONES 'Z'
#define UNITS 'U'

struct Matrix {
	double r1[NZ], hx[NZ+1], hm[NZ], parab_surf[NZ], hm4f[NZ], D1[NZ];
	double D2[NZ], D12[NZ], D12c[NZ], lamdac[NZ], lamdaf[NZ];
	double lamdafrho[NZ], u[NZ], wavefront[NZ+1];
	double deviation[NZ];
};
struct MirrorData {
	char *comments;
	double dia, roc, rho, cnst;
};
static struct best {
	int i, j, zone;
	double deviation, TrAbbr;
	double a, c;
};

struct Matrix M;
struct MirrorData Specs =  { NULL, 8.0, 96.0, 0., 0.};
static struct best best;

static double rdg[NZ][NRMAX];
float SimVal[NSIM];
float green_light = 21.6;

double nrand(double, double);
static void CalcZones(void);
static void calc(struct Matrix *m);
static void FmtMatrix(int, int, struct MirrorData *, struct Matrix *, FILE *);
static void PrFile(void);
static int cmp();

void best_focus_method(struct Matrix *m);

int nzones = 4;
char TesterType = 'f', DiaOpt = '1';
double factor = 1.0;
int nr;
int nsamp = 1;
int data_valid = 0;
char method = TEX_METHOD;
int _use_sphere = 0;	/* true if we compare against a sphere */

char *main_screen = {
		"	        FOUCAULT DATA REDUCTION  Version 2.2 \n"
		"   Copyright (c) 1995 by Larry Phillips and Dale Eason\n\n"
		"   d)  input mirror Data\n"
		"   c)  Calculate mirror\n"
		"   b)  Change center of curvature constant search method\n"
#ifdef DO_GRAPHICS
		"   g)  Graph results.\n"
#endif
		"   m)  Monte Carlo Analysis.\n"
		"   s)  Save mirror specs in a file.\n"
		"   a)  Output file for Aperture program.\n"
		"   u)  Select units of measurments \n"
		"   z)  Calculate Zone mask boundaries.\n"
		"\n"
		"   Q)  Quit\n"
};

char *method_menu = {
	"  Choose Center of curvature constant search method\n\n"
	"   T)   Texereau\n"
	"   E)   Exploratory focus\n"
	"\n\n\n"
	"Wave error computations can be done using either of two methods of finding\n"
	"the center of curvature constant 'c'.\n"
	"The default method uses a reference parabola as explained in Texereau.\n"
	"The other 'Exploratory Focus' is explained by Suiter in \"Telescope\n"
	"Making issue #32\". It searches for the constant in small increments and picks\n"
	"the constant that generates the least transverse aberation and wave error.\n"

};
char *mirror_menu = {
	"                            INPUT MIRROR CONSTANTS\n"
	"\n"
	"   f)  Input from file.\n"
	"   k)  Input from keyboard.\n"
	"   q)	abort\n"
	"\n"
	"   Mirror constants of:\n"
	"      Comments\n"
	"      Diameter\n"
	"      Radius of Curvature\n"
	"      Number of Zones\n"
	"      Readings per Zone\n"
	"      Tester type (moving or fixed light source).\n"
	"      Number of diameters\n"
	"      Center zone inside radius\n"
	"      Zone outside radiuses\n\n"
	"   Constants are entered from the keyboard or a file.\n"
	"   The file format is identical to the keyboard input one line\n"
	"   at a time.\n"
	"   You will be able to save the mirror constants in a file later on. \n"
	"\nSelect 'f' for file input, 'k' for keyboard input, or 'q' to abort.\n"
};

/* save aperture format wave error file */
void save_aper(void)
{
	char buf[RL];
	FILE *fp;
	int i;
	char file_name[RL] = "aper.dat";

	clrscr();
	puts("                 Write an Apeture File\n"
		 "Write a wave error file for input into the Apeture program written\n"
		 "by Dick Suiter.\n\n"
		 "Apeture is available from Willman-Bell, Inc.  It will compute performance\n"
		 "data for a telescope with this or any mirror as compared to a perfect\n"
		 "mirror.\n\n"
		 "It will also simulate a star image as it will appear at various\n"
		 "focus settings for this mirror.  It requires and IBM Compatable with\n"
		 "VGA graphics.\n\n\n");

	printf("File name [%s]", file_name);
	gets(buf);

	/* if other than default then store new file name */
	if (*buf)
		strcpy(file_name,buf);
	if ((fp = fopen(file_name, "wt")) == NULL) {
		printf("\nCan't open %s as save file.", file_name);
		return;
	}
	fprintf(fp, "%d\n",nzones);
	/* compute wave error at each outside zone edge */
	for (i = 0; i < nzones; i++) {
		double err;
		/* ref parabola */
		double y = best.a * M.hx[i] * M.hx[i] + best.c;
		double w;
		err = y - (M.wavefront[i]);
		w = err/green_light;
		fprintf(fp, "%6.2lf %7.4lf\n", 2. * M.hm[i]/Specs.dia,w);
	}
	fclose(fp);
}

/* save mirror constants in a file */
void save_specs(void)
{
	char buf[RL];
	FILE *fp;
	int i;
	char file_name[RL] = "mirror.dat";

	clrscr();
	printf("File name [%s]", file_name);
	gets(buf);

	/* if other than default then store new file name */
	if (*buf)
		strcpy(file_name,buf);
	if ((fp = fopen(file_name, "wt")) == NULL) {
		printf("\nCan't open %s as save file.", file_name);
		return;
	}

	if (Specs.comments)
		fprintf(fp, "%s", Specs.comments);
	else
		fprintf(fp, " ");
	fprintf(fp, "%6.2lf\n", Specs.dia);
	fprintf(fp, "%6.2lf\n", Specs.roc);
	fprintf(fp, "%d\n", nzones);
	fprintf(fp, "%d\n", nsamp);
	fprintf(fp, "%c\n", TesterType);
	fprintf(fp, "%c\n", DiaOpt);
	for ( i = 0; i <= nzones; i++)
		fprintf(fp, "%6.2lf\n", M.hx[i]);
	fclose(fp);
}

char prompt (char *str)
{
	clrscr();
	puts(str);
	return(getch());
}
#ifdef DO_GRAPHICS
/*
 * scale -      for scaling the axis of a plot. given the minimum and maximum
 *                      values in the data and a suggested number of intervals, the
 *                      returned structure defines a starting/ending value, distance
 *                      between tick marks, and the number of intervals (# ticks - 1).
 *                      adapted from the UNIX S "pretty" function
 */
struct scalev {
        int nint;                       // number of intervals
        int ndecimals;          // number of places to right of decimal
		double min, max;         // first, last value on the axis
		double incr;                     // distance between tick marks
};


#define NINT 5
#define ABS(A) ( (A) < 0.0 ? -(A) : (A) )


struct scalev *
scale(double umina, double umaxa, int na)
{
#       define DEL .0002
		static struct scalev s;
		static float vint[] = {1.0, 2.0, 5.0, 10.0};
		static float sqr[] = {1.414214, 3.162278, 7.071068};
        int n, i, nal, m1, m2;
        float umin, umax, dd, a, al, b, fn;
        float fm1, fm2;

		n = MAX(na, 1);
        umin = MIN(umina, umaxa);
		umax = MAX(umina, umaxa);
		if (umin == umax) {
				dd = ABS(umin);
                dd = MAX(0.25 * dd, 1.0e-2);
                umin = umin - dd;
                umax = umax + dd;
				n = 1;
		}
        fn = n;
		a = (umax - umin) / fn;
        al = log10((double) a);
        nal = al;
        if (a < 1.0)
                nal--;
        b = a / pow(10.0, nal);
        for (i = 0; i < 3; i++)
				if (b < sqr[i])
                        break;
		s.incr = vint[i] * pow(10.0, nal);

        if (s.incr < 0.00075)
                s.ndecimals = 4;
        else if (s.incr < 0.0075)
				s.ndecimals = 3;
        else if (s.incr < 0.075)
				s.ndecimals = 2;
        else if (s.incr < 0.75)
				s.ndecimals = 1;
        else
                s.ndecimals = 0;

        fm1 = umin / s.incr;
		m1 = fm1;
        if (fm1 < 0.0)
                m1--;
		dd = (float) m1 + 1.0 - fm1;
		if (ABS(dd) < DEL)
				m1++;
        s.min = s.incr * (float) m1;

        fm2 = umax / s.incr;
		m2 = fm2 + 1;
		if (fm2 < -1.0)
				m2--;
		dd = fm2 + 1.0 - (float) m2;
		if (ABS(dd) < DEL)
				m2--;
		s.max = s.incr * (float) m2;

		if (s.min > umin)
				s.min = umin;
		if (s.max < umax)
				s.max = umax;
		s.nint = ABS(m2 - m1);
		return &s;
}

int maxx, maxy;

/* initialize borland graphics */
/*  saves screen max x and y in globals maxx and maxy */
void open_graph(void)
{

   /* request auto detection */
   int gdriver = DETECT, gmode, errorcode;

	errorcode = registerbgidriver(EGAVGA_driver);

   /* initialize graphics mode */
   initgraph(&gdriver, &gmode, "");

   /* read result of initialization */
   errorcode = graphresult();

   if (errorcode != grOk)  /* an error occurred */
   {
	  printf("Graphics error: %s\n", grapherrormsg(errorcode));
	  printf("Press any key to halt:");
	  exit(1);             /* return with error code */
   }

	/* record screen limits */
	maxx = getmaxx();
	maxy = getmaxy();
}

/* draw lines between a list of points
 * input is:  pointer to array of x points
 *            pointer to array of y points
 *			  number of points
 *            xscale
 *            offset in screen pixels to be added x after scaling
 *            yscale
 *            offset in screen pixels to be added to y after scaling
 */
void plotxy(double *x, double *y, int num, double xscale,
			int xoffset,double yscale,  int yoffset)
{
	int i;
	for (i = 0; i < num-1; i++) {
		line(xoffset + x[i] * xscale, maxy - (yoffset + y[i] * yscale),
			 xoffset + x[i+1] * xscale, maxy - (yoffset + y[i+1] * yscale));
	}
}

/*  Find min and max values of an array of doubles.
 *  input is a pointer to the array, the number of values,
 *  pointers to where min and max results are stored
 *
 *   Returns max - min
 */
double find_range (double *val, int num, double *maxv, double *minv)
{
	int i;
	double minval = HUGE_VAL;
	double maxval = -HUGE_VAL;
	for (i = 0; i < num; i++) {
		minval = min(minval, val[i]);
		maxval = max(maxval, val[i]);
	}
	*maxv = maxval;
	*minv = minval;
	return (maxval - minval);
}
/* write vertical text */
void vert_text(int x, int y, char *str)
{
	int i;

	for (i = 0; str[i]; i++){
		char buf[2];
		buf[0] = str[i];
		buf[1] = 0;
		outtextxy(x, y + 10 * i, buf);
	}
}
/* make Millies-Lacroix tolerance plot of data in global M */
void make_ml_graph(int nzones)
{
	double maxdx, maxdy, mindx, mindy;
	double yscale;
	double xscale;
	double pointx[20];
	double pointy[20];
	double tmaxy, tminy;
	int xoffset = maxx/9;		/* left edge of plot window */
	int yoffset;
	int top = maxy /2 + 2;		/* top of plot window */
	int bottom = maxy - 2;		/* bottom of plot window */
	int right = maxx - 2;		/* right side of plot window */
	char *ytitle = {"Lambda C"};
	char buf[20];
	int i;

	double xstart = M.hm[0];
	double del;

	struct scalev *s;


	maxdx = Specs.dia/2l;
	xscale = (maxx - 10 - xoffset) / maxdx;

	/* fill plot with blue background */
	setfillstyle(SOLID_FILL,WHITE);
	/* left top right bottom */
	bar(10, top, right, bottom);

	setcolor(LIGHTGRAY);
	rectangle(10,top, right, bottom);

	bottom -= 48;		/* save room for x labels and title */
	top += 12;
	right -= 5;			/* leave a 5 pixel border at left of graph */

	/* calculate tolerance */
	del = (maxdx - M.hm[0])/9;
	for ( i = 0; i < 20; i++) {
		pointx[i] = xstart + del * (double)i;
		pointy[i] = 2. * Specs.roc * Specs.rho/pointx[i];
	}
	/* find the y range from the tolerance */
	find_range(pointy,10, &maxdy, &mindy);
	/* find the y range of the measurments */
	find_range(M.lamdac, nzones, &tmaxy, &tminy);


	/* if measurments have the larger values use them */
	if (tmaxy > maxdy)
		maxdy = tmaxy;
	if (tminy < mindy)
		mindy = tminy;


	/* if absolute min is larger than max use it as max*/
	if (fabs(tminy) > maxdy)
		maxdy = fabs(tminy);

	/* get y scale label increments */
	s = scale(maxdy, -maxdy, 15 );

	/* fit range into plot window */
	yscale = (bottom - top)/( s->max - s->min);

	settextjustify (RIGHT_TEXT, CENTER_TEXT);
	/* grid and lables are black */
	setcolor(BLACK);

	/* x axis will be in middle of plot window */
	yoffset = (top+bottom)/2;
	line (xoffset, yoffset, right, yoffset );

	/* draw y axis */
	line (xoffset, top, xoffset, bottom);

	/* label y axis and make tick marks */
	for (i = 0; i <= s->nint; i++) {
		double y = s->min + i * s->incr;
		int ypoint = yoffset -  yscale * y;

		/* label every other increment */
		if ((i&0x1) == 0) {
			sprintf(buf, "%5.*lf", s->ndecimals, y);
			outtextxy(xoffset - 3, ypoint, buf);
			line(xoffset , ypoint, xoffset+5, ypoint);
		}
		else {
			line(xoffset, ypoint, xoffset+3, ypoint);
		}
	}

	/* Vertical title for y */
	/* y title start a middle - 1/2 title length */
	vert_text(xoffset-50, (top + bottom) /2 - strlen(ytitle) * 5, ytitle);

	setcolor(BLACK);
	setlinestyle(SOLID_LINE,0,3);
	plotxy(M.hm, M.lamdac, nzones, xscale, xoffset, yscale, maxy - yoffset);

	setlinestyle(SOLID_LINE,0,1);
	setcolor(CYAN);
	plotxy(pointx, pointy, 10, xscale, xoffset, yscale, maxy - yoffset);

	for (i = 0; i < 10; i++) {
		pointy[i] = -pointy[i];
	}
	plotxy(pointx, pointy, 10, xscale, xoffset, yscale, maxy - yoffset);


	/* draw bottom line of graph */
	setcolor(BLACK);
	line(xoffset, bottom, maxx - 10, bottom);

	/* label x axis */
	settextjustify(CENTER_TEXT, TOP_TEXT);
	bottom += 12;
	setlinestyle(DOTTED_LINE,0,1);
	for (i = 0; i < nzones; i++ ) {
		int x = xoffset + xscale * M.hm[i];
		sprintf(buf, "%6.2lf", M.hm[i]);
		outtextxy( x, bottom, buf);
		line (x, bottom - 2, x, top);
	}
	bottom += 12;
	outtextxy( (xoffset + right) /2, bottom, "Zone Center");
	bottom += 12;
	outtextxy( (xoffset + right)/2, bottom, "Millies-Lacroix Tolerance "
				"(parabola removed)");
	setlinestyle(SOLID_LINE,0,1);
}

/* make surface error graph of data in global M */
void make_surface_graph(int nzones)
{
	double maxdx, maxdy, mindx, mindy;  /* min and max values of data */

	double yscale;
	double xscale;
	double pointy[NZ];		/* temp y points */
	double range;

	int i;
	int xoffset = maxx/9;		/* leave room on left side for y labels */
	int yoffset;
	/* use the upper half of the screen */
	int bottom = maxy/2  - 2;	/* bottom edge of plot window */
	int top = 2;                /* top edge of plot window */
	int right = maxx - 2;	    /* right side of plot window */
	char *ytitle = "Wavelength";
	char buf[20];

	/* fill plot with blue background */
	setfillstyle(SOLID_FILL,WHITE);
	/* left top right bottom */
	bar(10, top, right, bottom);

	/* draw a white border */
	setcolor(LIGHTGRAY);
	rectangle(10,top, right, bottom);

	/* the radius is the max x value*/
	maxdx = Specs.dia/2.;

	/* leave a 5 pixel boarder inside plot window */
	right -= 5;
	/* set x scale.  Leave a 10 pixel boarder on the right side */
	xscale = (right - xoffset )/maxdx;

	for (i = 0; i < nzones+1; i++)
		pointy[i] = M.wavefront[i];

	range =  find_range(pointy, nzones+1, &maxdy, &mindy);
	if (range < 2.)
		range = 2.;

	/* leave a boader at top of 12 pixels */
	top += 12;

	/* leave a boarder at the bottom for labels */
	bottom -= 44;

	/* draw x grid zone lines */
	setcolor(BLACK);
	setlinestyle(DOTTED_LINE,0,1);
	for ( i = 0; i <= nzones; i++)
		line(xoffset + M.hx[i] * xscale, top, xoffset + M.hx[i] * xscale,
			 bottom);

	yscale = (bottom - top) / ( 2. * range);
	yoffset = (bottom + top)/2;
	setcolor(BLACK);
	setlinestyle(SOLID_LINE,0,3);
	plotxy(M.hx, pointy, nzones+1, xscale, xoffset, yscale, maxy - yoffset);

	setlinestyle(SOLID_LINE,0,1);

	setcolor(BLACK);
	/* draw y axis */
	line(xoffset, yoffset, right, yoffset);

	/* draw top line of graph */
	line (xoffset, top, right,top);

	/* draw bottom line of graph */
	line (xoffset, bottom, right,bottom);

	/* label graph */

	/* y labels */
	settextjustify(RIGHT_TEXT,CENTER_TEXT);
	sprintf(buf, "%6.2lf", range/green_light);
	outtextxy( 70 , top , buf);
	sprintf(buf, "-%6.2lf", range/green_light);
	outtextxy( 70 , bottom, buf);
	outtextxy(70, yoffset,"0");


	/* Vertical title for y */
	/* y title start a middle - 1/2 title length */
	vert_text(xoffset - 30,(top + bottom)/2 - ( strlen(ytitle) * 5), ytitle);

	bottom += 10;                /* move bottom to first x label line */

	/* label the x zone values */
	for (i = 0; i < nzones; i++) {
		/* x labels */
		settextjustify(CENTER_TEXT, TOP_TEXT);
		sprintf(buf, "%6.2lf", M.hx[i]);
		outtextxy (xoffset + M.hx[i] * xscale, bottom ,buf);
	}


	/* x axis label */
	/* center the following text */
	settextjustify(CENTER_TEXT, TOP_TEXT);
	bottom += 10;		/* move bottom down 1 text line */
	outtextxy(maxx/2, bottom, "Radius");

	/* title */
	bottom += 10;		/* move bottom douwn 1 text line */
	outtextxy(maxx/2, bottom, "RELATIVE SURFACE ERROR (+ surface too high, - surface too low");

}
void draw_histo(int size, int *histo, int maxval,int ten_percent, int ninty_percent)
{

	double yscale;

	int i;
	int xoffset = 70;       /* leave room on left side for y labels */
	int yoffset;

	/* use the upper half of the screen */
	int bottom = maxy/2  +40;   /* bottom edge of plot window */
	int top = 2;                /* top edge of plot window */
	int right = maxx - 2;       /* right side of plot window */
	char buf[20];
	int del;
	int space;
	struct scalev *s;



	/* fill plot with blue background */
	setfillstyle(SOLID_FILL,WHITE);
	/* left top right bottom */
	bar(10, top, right, bottom);

	/* draw a white border */
	setcolor(LIGHTGRAY);
	rectangle(10,top, right, bottom);
	right -= 15;
	del = (right - xoffset)/ size;
	space = del /5;
	bottom -= 55;
	top += 18;

	/* get y scale label increments */
	s = scale(0, maxval, 8 );

	yscale = (double)(bottom - top )/s->max;

	/* draw  axis */
	setcolor(BLACK);
	line (xoffset, top, xoffset, bottom);
	line (xoffset, bottom, right, bottom);

	settextjustify(RIGHT_TEXT, CENTER_TEXT);
	/* label y axis and make tick marks */
	for (i = 1; i <= s->nint; i++) {
		double y = s->min + i * s->incr;
		int ypoint = bottom -  yscale * y;

		/* label every other increment */
		if ((i&0x1) == 0) {
			sprintf(buf, "%5.*lf", s->ndecimals, y);
			outtextxy(xoffset - 10, ypoint, buf);
			line(xoffset -5, ypoint, right, ypoint);
		}
		else {
			line(xoffset-3, ypoint, xoffset, ypoint);
		}

	}
	/* Vertical title for y */
	strcpy(buf, "Simulations");
	vert_text(xoffset - 40,  (top + bottom) /2 - ( strlen(buf) * 6), buf);

	settextjustify(CENTER_TEXT, TOP_TEXT);
	/* draw bar graph */
	for (i = 0; i < size; i++) {
		int x1 = xoffset + i * del + space;
		int x2 = x1 + 3 * space;
		/*  highlight the middle 80 % */
		if ((i < ten_percent) || (i > ninty_percent))
			setfillstyle(SOLID_FILL, LIGHTGRAY);
		else
			setfillstyle(SOLID_FILL, RED);
		bar(x1, bottom - yscale * histo[i], x2, bottom);
		rectangle(x1, bottom - yscale * histo[i], x2, bottom);

		/* label the x axix */
		if (i == 0)
			strcpy(buf, ">1");
		else if (i == size-1)
			sprintf(buf, "<1/%d",i);
		else
			sprintf(buf, "1/%d", i);

		outtextxy((x1+x2)/2, bottom + 10 + 12 * (i&0x1), buf);
		/* draw x axis tick marks */
		line(x1 - space, bottom -3, x1 -space, bottom +3);
	}
	bottom += 40;
	settextjustify(CENTER_TEXT, TOP_TEXT);
	outtextxy(right/2, bottom, "Wave rating. (RED is 80% Confidence section.)");
	outtextxy(right/2, top - 15, "monte CARLO SIMULATION HISTOGRAM");
}
#endif
char	*monte_descr = {
		"The variance of the measurments determines the confidence of the\n"
		"results.  This simulation will jitter each mean zone reading by\n"
		"plus and minus a random amount based on the standard deviation of your\n"
		"readings.  If you entered one set of readings the deviation\n"
		"is set at a default of .005.  You can change the deviation settings\n"
		"now.  Up to 8000 simulations can be made using these random readings\n"
		"If your computer is slow you can reduce the number of simulations.\n"
		"The more simulations the more accurate the 80% confidence\n"
		"will be.  The results will be displayed in a fractions of wavelength\n"
		"wave error histogram and the 80% confidence level will be shown."
};

void monte(int nr, int nzones) {
	int i;
	int j;
	double zone_mean[NZ], zone_stdev[NZ], t, tssq;
#define BUCKETS  24
	int  histo[BUCKETS];		/* histogram */
	double bucket;
	double min  = HUGE_VAL;
	double max  = -HUGE_VAL;
	char buf[100];
	int max_bucket = 0;
	int max_sims = 1000;
	int meter;
	int percent;
	double trdg[NZ][NRMAX];
	struct Matrix tm = M;

	for (i = 0; i < BUCKETS; i++)
		histo[i] = 0;

	for (i = 0; i < nr; i++)
		for (j = 0; j < nzones; j++)
			trdg[j][i] = 0.0;

	/*
	 * do monte carlo simulation of additional sets of readings to determine
	 * what range of accuracy would be produced...
	 */

	if (nr > 1) {
		for (i=0; i < nr; i++) {
			/* make the mean of all the readings in each set equal to 1.0... */
			t = 0;
			for (j=0; j < nzones; j++)
				t += rdg[j][i];
			t /= nzones;
			for (j=0; j < nzones; j++)
				trdg[j][i] = rdg[j][i] - (t - 1.0);
		}
		for (i=0; i < nzones; i++) {
			/* mean & st dev of readings for each zone... */
			t = tssq = 0;
			for (j=0; j < nr; j++) {
				t += trdg[i][j];
				tssq += trdg[i][j] * trdg[i][j];
			}
			zone_mean[i] = t / nr;
			zone_stdev[i] = sqrt( (nr * tssq - t * t) / (nr * (nr - 1)) );
		}

	}
	else {		/* the mean is the value and stdev is .005 */
		for (i = 0; i < nzones; i++) {
			zone_mean[i] = rdg[i][0];
			zone_stdev[i] = .005;
		}
	}

	do {
		clrscr();
		puts("              monte Carlo Analysis\n");
		puts(monte_descr);
		puts("\n\n\n");
		for (i = 0; i < nzones; i++) {
			printf("Deviation for Zone %d [%6.4lf] ", i, zone_stdev[i]);
			gets(buf);
			if (*buf == 0)
				continue;
				zone_stdev[i] = atof(buf);
		}
		puts("Are these correct (y or n)? ");
		gets(buf);
	} while (*buf == 'n');

	sprintf(buf, "%d", max_sims);
	printf("Number of simulations. (%d max) [%s] ", NSIM, buf);
	gets(buf);
	if (*buf != 0)
		max_sims = atoi(buf);

	if (max_sims > NSIM)
		max_sims = NSIM;

	if (max_sims < 20)
		max_sims = 20;

	percent = max_sims/20;
	if (percent < 1)
		percent = 1;

	printf("Hit any key to quit early.\n");
	printf("         %-4d      %4d\n", max_sims/2, max_sims);
	puts( "|.........|.........|");
	putchar('+');
	for (i=0, meter = percent; i < max_sims; i++) {

		/* simulate NSIM sets of readings, and compute accuracy for each... */
		for (j=0; j < nzones; j++)
			tm.D12[j] = nrand(zone_mean[j], zone_stdev[j]);

		if (method == 1)
			best_focus_method(&tm);
		else
			calc(&tm);

		SimVal[i] = green_light / best.deviation;
		if (SimVal[i] < min)
			min = SimVal[i];
		if (SimVal[i] > max)
			max = SimVal[i];

		j = SimVal[i];
		if (j >= BUCKETS)
			j = BUCKETS -1;
		histo[j]++;

		/* record the largest bucket */
		if (histo[j] > max_bucket)
			max_bucket = histo[j];

		if (i >= meter) {
			putchar('+');
			meter += percent;
		}
		if (fast_kbhit()) {
			max_sims = i;
			getch();
			break;
        }
    }
#ifndef DO_GRAPHICS
	printf("\n > 1  ", histo[0]);
	for (i = 1; i < BUCKETS-1; i++)
		printf("\n1/%2d|  %d", i, histo[i]);
	printf("\n<1/%2d %d",i,histo[i]);
#endif
	qsort(SimVal, max_sims, sizeof(SimVal[0]), cmp);
	i = 10. * (double)max_sims / 100.;
	j = 89. * (double)max_sims / 100.;

#ifndef DO_GRAPHICS
	printf( "    80%% Confidence Limits: 1 / %4.1f to 1 / %6.2f\n",
		SimVal[i], SimVal[j]);
	getch();
#else

	/* turn i and j into histogram indexes */
	i = SimVal[i];
	j = SimVal[j];
	if (j >= BUCKETS)
		j = BUCKETS -1;

	open_graph();
	draw_histo(BUCKETS, histo, max_bucket, i,j);
	getch();
	closegraph();
#endif

}
void make_data_table(void)
{
	FILE *fpt;
	int i;
	int j;

	Specs.rho = 1.22 * green_light * 1e-6 * Specs.roc / Specs.dia / 2.;

	if (method == TEX_METHOD)
		calc(&M);
	else
		best_focus_method(&M);

	if ( (fpt = fopen(TXT, "w") ) == NULL) {
		printf("Open failed for file %s\n", TXT);
		return;
	}
	FmtMatrix(nzones, nsamp, &Specs, &M, fpt);
	if (nr > 1) {
		struct Matrix tm = M;
		fprintf(fpt, "\nResults for individual sets of readings:\n");
		for (i=0; i < nr; i++) {
			for (j=0; j < nzones; j++)
			   tm.D12[j] = rdg[j][i];
			if (method == TEX_METHOD)
				calc(&tm);
			else
				best_focus_method(&tm);

			fprintf(fpt, "\tReading set %d: 1 / %4.1f lamda, "
				"Max Trans Abberation=%2.2f\n", i+1,
				green_light / best.deviation, (float) best.TrAbbr);
		}
	}
	fclose(fpt);
}
void input_mirror_data(void)
{
	int nresp;
	char **resp;
	char buf[RL];
	double val;
	int i,j,k;
	double t;
	int ndia = 1;
	int cmd;

	FILE *fmirror = NULL;

	/* data entry... */
	printf("Texereau Mirror Test Program\n\n");


	do {
		cmd = toupper(prompt(mirror_menu));
		switch (cmd) {
			char file_name[RL];
			case MIRROR_FILE:
				clrscr();
				sprintf(file_name, "mirror.dat");
				printf("\nMirror data file name? [%s] ",file_name);
				gets(buf);
				if (*buf)
					strcpy(file_name, buf);
				if ((fmirror = fopen(file_name, "rt")) == NULL) {
					printf("\nCan't open %s as mirror data file", file_name);
					getch();
					continue;
				}
				cmd = QUIT;
				break;

			case 'Q':
				return;

			case KEYBOARD:
				fmirror = stdin;
				cmd = QUIT;
				break;
			default:
				break;
		}
	} while (cmd != QUIT);

	do {
		sprintf(buf, "%s", Specs.comments);
		printf("Comments: [%s] ", buf);
		fgets(buf, RL, fmirror);
		if (*buf != '\n') {
			if (Specs.comments)
				free(Specs.comments);
			Specs.comments = strdup(buf);
		}
		if (fmirror != stdin)
			printf("%s", Specs.comments);
		sprintf(buf,"%6.2lf", Specs.dia);
		printf("Optical diameter of mirror [%s] ", buf);
		fgets(buf, RL, fmirror);
		if (*buf != '\n')
			val = Specs.dia = atof(buf);
		if (fmirror != stdin)
			printf("%6.2lf", Specs.dia);

		sprintf(buf, "%6.2lf", Specs.roc);
		printf("\nRadius of curvature: [%s] ", buf);
		fgets(buf, RL, fmirror);
		if (fmirror != stdin)
			printf(" %s",buf);

		if (*buf != '\n')
			val = Specs.roc = atof(buf);


		sprintf(buf, "%d", nzones);
		printf("Number of mask zones (%d max):  [%s] ",NZ, buf);

		fgets(buf, RL, fmirror);
		if (*buf != '\n')
			nzones = atoi(buf);

		if (nzones < 1)
			nzones = 1;

		nzones = min(nzones, NZ);

		if (fmirror != stdin)
			printf(" %d\n",nzones);

		sprintf(buf, "%d", nsamp);
		printf("Readings per zone (%d max) : [%s] ",NRMAX/2, buf);
		fgets(buf, RL, fmirror);


		if (*buf != '\n')
			nsamp = atoi(buf);

		if (nsamp < 0)
			nsamp = 1;

		if (nsamp > NRMAX/2)
			nsamp = NRMAX/2;

		if (fmirror != stdin)
			printf(" %d\n",nsamp);

		sprintf(buf,"%c", TesterType);
		printf("Is the light source fixed, or "
			"does it move with the knifedge (f or m): [%s] ", buf);
		fgets(buf, RL, fmirror);
		if (fmirror != stdin)
			printf(" %s",buf);

		if (*buf != '\n')
			TesterType = tolower(*buf);
		if (TesterType == 'm') {
			factor = 2.0;
			printf("Moving source tester; readings will be shown doubled\n");
		}
		else
			factor = 1.0;

		sprintf(buf, "%c", DiaOpt);
		printf("Readings taken on 1 or 2 diameters: [%s] ", buf);
		fgets(buf, RL, fmirror);

		if (fmirror != stdin)
			printf(" %s",buf);

		if (*buf != '\n')
			DiaOpt = *buf;

		ndia = (DiaOpt == '2' ? 2 : 1);

		sprintf(buf, "%6.2lf", M.hx[0]);
		printf("Inside radius of zone 1 (may be 0) [%s] ", buf);
		fgets(buf, RL, fmirror);
		if (*buf != '\n')
			M.hx[0] = atof(buf);
		if (fmirror != stdin)
			printf("%s", buf);
		nresp = 1;
		for (i=1; i <= nzones; i++) {
			sprintf(buf, "%6.2lf", M.hx[i]);
			printf("Outside Radius of zone %d: [%s] ", i, buf);
			fgets(buf, RL, fmirror);
			if (fmirror != stdin)
				printf(" %s", buf);

			if (*buf != '\n')
				M.hx[i] = atof(buf);
			M.hm[i-1] = (M.hx[i] + M.hx[i-1]) / 2;	/* average zone width */
			if (_use_sphere)
				// longitudinal deviation for a sphere is zero
				M.parab_surf[i-1] = 0;
			else
				// theoretical longitudinal deviation (fixed source tester)
				M.parab_surf[i-1] = M.hm[i-1] * M.hm[i-1] / Specs.roc;
			M.hm4f[i-1] = M.hm[i-1] / (2 * Specs.roc);
		}
		/* switch to keyboard input  for corrections */
		if (fmirror != stdin) {
			fclose(fmirror);
			fmirror = stdin;
		}
		printf("\nAre the above values correct (y or n)?: ");
		gets(buf);
	} while (*buf == 'n');


	// read in the knife edge readings...
	do {		// loop until readings are acceptable
		nresp = 0;
		for (i=0; i < ndia; i++) {
			printf("\n\n");
			if (DiaOpt == '2')
				printf("Enter readings taken on radius %d\n", i+1);
			for (j = 0; j < nzones; j++) {
				printf("\tZone %d\n", j+1);
				t = 0.;
				nr = i * nsamp;
				for (k = 0; k < nsamp; k++) {
					sprintf(buf, "%6.4lf", rdg[j][nr]/factor);
					printf ("\t\tReading #%d [%s] ", k+1, buf);
					gets(buf);
					val = rdg[j][nr]/factor;
					if (*buf)
					   val = atof(buf);
					nresp++;
					val *= factor;
					t += val;
					rdg[j][nr++] = val;
				}
				if (i == 0) {
					M.D1[j] = t / nsamp;
					if (DiaOpt == '1')
						M.D2[j] = M.D1[j];
				}
				else
					M.D2[j] = t / nsamp;
			}
		}
		printf("\nAre the above readings correct (y or n)?: ");
		gets(buf);
	} while (*buf == 'n');

	for (i=0; i < nzones; i++)
		/* average the D1 and D2 readings*/
		M.D12[i] = (M.D1[i] + M.D2[i]) / 2;

	data_valid = 1;
}
void data_invalid(void)
{
	clrscr();
	puts("You must input the mirror constants first.");
	getch();
}

void calc_wavefront(struct Matrix *m, double coc, double *dmin, double *dmax, double *trab)
{
	int i;
	double min = +HUGE_VAL;
	double max = -HUGE_VAL;
	double maxtrab = -HUGE_VAL;
	double tedge;

	m->wavefront[0] = 0;
	for (i = 0; i <= nzones; i++) {
		m->lamdac[i] = m->D12[i] - m->parab_surf[i] - coc;
		m->lamdafrho[i] = m->lamdac[i] * m->hm[i]/ (2. * Specs.roc * Specs.rho);

		if (fabs(m->lamdafrho[i]) > maxtrab)
			maxtrab = fabs(m->lamdafrho[i]);

		m->u[i] = -m->lamdafrho[i] * Specs.rho * 1e6/(Specs.roc / 2.);

		if (i > 0)
			m->wavefront[i] = (m->hx[i] - m->hx[i-1]) * m->u[i-1] +
			m->wavefront[i-1];

		if (m->wavefront[i] > max)
			max = m->wavefront[i];
		if (m->wavefront[i] < min)
			min = m->wavefront[i];
	}
	tedge = m->lamdafrho[nzones-2] + (m->hx[nzones-1] - m->hm[nzones-2]) *
			(m->lamdafrho[nzones-1] - m->lamdafrho[nzones-2]) /
			(m->hm[nzones-1] - m->hm[nzones-2]);

	if (tedge > maxtrab)
		maxtrab = tedge;
	*dmax = max;
	*dmin = min;
	*trab = maxtrab;
}
void best_focus_method(struct Matrix *m)
{
	char buf[RL];
	int i;
	double coc;
	double min = + HUGE_VAL;
	double max = -HUGE_VAL;
	double v;
	double first, last, step;
	double c;
	double trab;
	double best_coc;
	double best_trab = HUGE_VAL;
	double best_trab_coc;
	double best_wave = HUGE_VAL;
	double best_wave_coc = HUGE_VAL;
	double wave;

	FILE *fpt;

	v = 2. * Specs.rho * Specs.roc / m->hm[nzones-1];

	c = (m->D12[nzones-1] - m->parab_surf[nzones-1] +
			m->D12[nzones - 2] - m->parab_surf[nzones - 2]) / 2.;

	first = c  - 2. * v;
	last = c + 2. * v;
	step = v/20.;

	for (coc = first; coc <= last; coc += step) {
		calc_wavefront(m, coc, &min, &max, &trab);
		wave = max - min;

		/* find best wave when trab is less than 1. */
		if (trab < 1.) {
			if (wave < best_wave) {
				best_wave = wave;
				best_wave_coc = coc;
			}
		}
		if (trab < best_trab) {
			best_trab = trab;
			best_trab_coc = coc;
		}
	}
	/* if best transvers is less than 1. then use best wave coc */
	if (best_trab < 1.) {
		best_coc = best_wave_coc;
	}
	/* esle use best trab coc */
	else {
		best_coc = best_trab_coc;
	}

	calc_wavefront(m, best_coc, &min, &max, &trab);
	best.deviation = max - min;
	max = -HUGE_VAL;
	/* find zone of max deviation */
	for (i = 0; i < nzones; i++) {
		double val = fabs(m->wavefront[i]);
		if (val > max) {
			max = val;
			best.zone = i;		/* zone with worst deviation */
		}
	}
	best.a = 0.;
	best.c = best_coc;
	Specs.cnst = best_coc;
}
static int  far *head_keys = (int far *) 0x0040001aL;
static int far *tail_keys = (int far *) 0x0040001cL;
int	fast_kbhit(void)
{
	if (*head_keys != *tail_keys)
		return(1);
	else
		return(0);
}

void change_units(void)
{
	char c;
	int i;

	puts("C for centimeters, M for millimeters, or I for Inches");
	c = getch();


	switch (toupper(c)) {
		case 'C':
			green_light = 54.87;
			break;
		case 'M':
			green_light = 548.7;
			break;
		case 'I':
			green_light = 21.6;
			break;
	}
}

main(int argc, char **argv)
{
	int  nsamp, i, j, k;
	float val;
	double factor;
	double t;
	FILE *fpt;
	char cmd;


	if (argc > 1 && strcmp(argv[1], "-m") == 0) {
		CalcZones();
		return 0;
	}
	if (argc > 1 && strcmp(argv[1], "-s") == 0)
		_use_sphere = 1;

//	input_mirror_data();
//	if (data_valid) {
//		PrFile();
//		getch();
//	}

	do {
		cmd = toupper(prompt(main_screen));
		switch (cmd) {
			int tmp;
			case METHOD:
				tmp = prompt(method_menu);
				switch (toupper(tmp)) {
					case TEX_METHOD:
						method = TEX_METHOD;
						break;
					case EF_METHOD:
						method = EF_METHOD;
						break;
				}
				if (data_valid) {
					make_data_table();
					PrFile();
					getch();
				}
				break;

			case MIRROR_DATA:
				input_mirror_data();
				if (data_valid) {
					make_data_table();
					PrFile();
					getch();
				}
				break;
			case CALC_TABLE:
				if (!data_valid) {
					data_invalid();
					break;
				}
				make_data_table();
				PrFile();
				getch();
				break;
			case MONTY_CARLO:
				if (!data_valid) {
					data_invalid();
					break;
				}
				monte(nr, nzones);
				break;
			#ifdef DO_GRAPHICS
			case GRAPHS:
				if (!data_valid) {
					data_invalid();
					break;
				}
				open_graph();
				make_surface_graph(nzones);
				make_ml_graph(nzones);
				getch();
				closegraph();
				break;
			#endif
			case APETURE_FILE:
				if (!data_valid) {
					data_invalid();
					break;
				}
				save_aper();
				break;
			case QUIT:
				break;
			case ZONES:
				CalcZones();
				getch();
				break;

			case SAVE_SPECS:
				if (!data_valid) {
					data_invalid();
					break;
				}
				save_specs();
				break;

			case UNITS:
				change_units();
				break;

			default:
				break;
		}
	} while (cmd != QUIT);
	return 0;
}
int
cmp(float *a1, float *a2)
{
	if (*a1 > *a2)
		return 1;
	else if (*a1 == *a2)
		return 0;
	else
		return -1;
}

static void calc(struct Matrix *tm)
{
	double dev, MinAbr, MaxAbr, m, b, disc;
	double diff, x1, y1, x2, y2, a, c, y, t, maxdev, tdev;
	int i, j, k, type, maxzone;

	/* compute "suitable constant" cnst to be subtracted from the averaged
	 * values so that max and min deviations are equal and opposite in sign...
	 */
	tdev = Specs.cnst = 0;
	do {
		Specs.cnst += tdev / 150000;
		for (i = 0; i < nzones; i++)
			tm->lamdaf[i] = tm->hm4f[i] * (tm->D12[i] - tm->parab_surf[i] - Specs.cnst) * 100000;

		MinAbr = 1e+10;
		MaxAbr = -MinAbr;
		for (i=0; i < nzones; i++) {
			MaxAbr = MAX(MaxAbr, tm->lamdaf[i]);
			MinAbr = MIN(MinAbr, tm->lamdaf[i]);
		}
		x1 = MinAbr + MaxAbr;
		if (tdev > 0 && fabs(x1) > fabs(tdev)) {
			printf("Search for constant failed; program terminated\n");
			printf("Check your input for sanity\n");
			exit(0);
		}
		tdev = x1;
	} while (fabs(tdev) > .001);

	best.TrAbbr = -1e5;
	for (j=0; j < nzones; j++) {
		tm->D12c[j] = tm->D12[j] - Specs.cnst;			/* adjust readings by cnst */
		tm->lamdac[j] = tm->D12c[j] - tm->parab_surf[j];	/* residual abberation at COC */
		/* transverse abberation compared with the Airy disk. To be plotted
		 * against average radii (hm) for the upper graph...*/
		tm->lamdafrho[j] = (tm->lamdaf[j] / 100000) / Specs.rho;
		if (fabs(tm->lamdafrho[j]) > best.TrAbbr)
			best.TrAbbr = fabs(tm->lamdafrho[j]);
		tm->u[j] = -tm->lamdaf[j] * 2 * 10 / Specs.roc;
	}

	/* graph 2 values; wavefront deviations... */
	tm->wavefront[0] = 0;
	for (i=1; i <= nzones; i++)
		tm->wavefront[i] = tm->wavefront[i-1] + tm->u[i-1] * (tm->hx[i] - tm->hx[i-1]);

	/* find the best reference parabola...
	 */
	best.deviation = 1e5;
	for (i = 0; i < nzones; i++) {	/* try parabola thru each pair of points */
		for (j = i + 1; j <= nzones; j++) {
			x1 = tm->hx[i], y1 = tm->wavefront[i];
			x2 = tm->hx[j], y2 = tm->wavefront[j];
			/* parabola equation: y = a * x**2 + c */
			a = (y2-y1)/(x2*x2 - x1*x1);
			c = y1 - a * x1 * x1;
			/* reject this parabola if points are on opposite sides of it */
			type = NEUTRAL;

			for (k=0; k <= nzones; k++) {
				y = a * tm->hx[k] * tm->hx[k] + c;
				diff = y - tm->wavefront[k];
				if (fabs(diff) < 0.0001)
					continue;
				switch (type) {
					case BELOW:
						if (diff > 0)
							goto NextPara;
						break;
					case ABOVE:
						if (diff < 0)
							goto NextPara;
						break;
					case NEUTRAL:
						type = (diff > 0 ? ABOVE : BELOW);
						break;

				}
			}
			/* if the 2 points are consecutive and ( (parabola curves up and
			 * other points are below it) or (parabola curves down and other
			 * points are above) ) */
			if (j == i+1 &&
			  ( (a > 0 && type == ABOVE) || (a < 0 && type == BELOW) ) )
				goto NextPara;
			/* reject if line segment intersects the parabola */
			for (k=0; k < nzones; k++) {
				m = tm->u[k];
					/* segment eqn: y=m*x + b... */
				b = tm->wavefront[k] - tm->u[k] * tm->hx[k];
				disc = m * m - 4 * a * (c - b);
				if (disc < 0)
					continue;	/* line does not intersect the parabola */
				x1 = (m - sqrt(disc)) / (2 * a);
				x2 = (m + sqrt(disc)) / (2 * a);
				if (x1 > x2) {
					t = x1;
					x1 = x2;
					x2 = t;
				}
				if (x1 > (tm->hx[k] +0.001) && x2 < (tm->hx[k+1] - 0.001) )
					goto NextPara;	/* segment intersects parabola */
			}
			/* find maximum deviation from this parabola... */
			maxdev = -1e5;
			for (k=0; k <= nzones; k++) {
				if (k == i || k == j)
					continue;
				y1 = a * tm->hx[k] * tm->hx[k] + c;
				dev = fabs(y1 - tm->wavefront[k]);
				if (dev > maxdev) {
					maxdev = dev;
					maxzone = k;
				}
			}
			if (maxdev < best.deviation) {
				best.deviation = maxdev;	/* best parabola found so far */
				best.i = i;
				best.j = j;
				best.a = a;
				best.c = c;
				best.zone = maxzone;
			}
			NextPara:
			;
		}
	}
	/* subtract ref parabola from wave front */
	for (k = 0; k <= nzones; k++ ) {
		y1 = best.a * tm->hx[k] * tm->hx[k] + best.c;
		tm->wavefront[k] -= y1;
	}
}

void
CalcZones()
{
	int nzones, i;
	char type;
	float val;
	double dia, AreaPerZone, center, OR, IR;

	printf("Compute dimensions of aperatures for the Couder Mask\n\n");
	printf("Two methods of making the mask:\n");
	printf("  a) intermost zone (zone 1) is a single opening that includes "
		"the center\n");
	printf("  b) intermost zone has 2 openings, with a circular area at the "
		"center\n     masked off\n");
	printf("Enter choice (a or b): ");
	scanf("%1s", &type);
	printf("Enter number of zones: ");
	scanf("%d", &nzones);
	printf("Enter optical diameter of mirror: ");
	scanf("%f", &val), dia = val;
	if (type == 'b') {
		printf("Enter radius of central masked area: ");
		scanf("%f", &val), IR = val;
		center = IR * IR;
	}
	else
		IR = center = 0;
	AreaPerZone = (dia * dia / 4 - center) / nzones;
	printf("\nZone  IR     OR     ID      OD\n");
	for (i = 1; i <= nzones; i++) {
		OR = sqrt(center + i * AreaPerZone);
		printf("%2d  %5.2f  %5.2f  %5.2f   %5.2f\n",
			i, IR, OR, 2 * IR, 2 * OR);
		IR = OR;
	}
	return;
}

void
FmtMatrix(int nzones, int nsamp, struct MirrorData *S,
	struct Matrix *M, FILE *fpt)
{
	int i;

	fprintf(fpt, "\n\tTEXEREAU MIRROR TEST SHEET\n\n");
	fprintf(fpt, "           Comments: %s", S->comments);
	fprintf(fpt, "   Optical diameter: %g\n", (float) S->dia);
	fprintf(fpt, "  Readings per zone: %d\n", nsamp);
	fprintf(fpt, "Radius of curvature: %g\n", (float) S->roc);
	fprintf(fpt, "                f/D: %5.2f\n", (float) S->roc / S->dia / 2);
	fprintf(fpt, "   Diffraction disc: %g\n", (float) S->rho);
	fprintf(fpt, "\n 1 %-16s", "ZONE");
	for (i=1; i <= nzones; i++)
		fprintf(fpt, "    %-5d  ", i);
	fprintf(fpt, "\n 2 %-13s", "h(x)");
	for (i=1; i <= nzones; i++)
		fprintf(fpt, "%11.4f", M->hx[i]);
	fprintf(fpt, "\n 3 %-13s", "h(m)");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->hm[i]);
	fprintf(fpt, "\n 4 %-13s", "hm**2/R");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->parab_surf[i]);
	fprintf(fpt, "\n 5 %-13s", "hm/4f");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->hm4f[i]);
	fprintf(fpt, "\n 6 %-13s", "D1");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->D1[i]);
	fprintf(fpt, "\n 7 %-13s", "D2");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->D2[i]);
	fprintf(fpt, "\n 8 %-13s", "D12");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->D12[i]);

	if (S->cnst <= 0.)
		fprintf(fpt, "\n 9 D12 + %7.4f", (float) fabs(S->cnst));
	else
		fprintf(fpt, "\n 9 D12 - %7.4f", (float) S->cnst);

	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->D12[i] - S->cnst);
	fprintf(fpt, "\n10 %-13s", "Lamda c");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.4f", M->lamdac[i]);
	fprintf(fpt, "\n11 %-13s", "Lamda f * 1e5");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.2f", M->lamdaf[i]);
	fprintf(fpt, "\n12 %-13s", "Lamda f / rho");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.3f", M->lamdafrho[i]);
	fprintf(fpt, "\n13 %-13s", "u * 1E6");
	for (i=0; i < nzones; i++)
		fprintf(fpt, "%11.2f", M->u[i]);
	fprintf(fpt, "\n14 %-13s", "Wavefront");
	for (i=1; i <= nzones; i++)
		fprintf(fpt, "%11.2f", M->wavefront[i]);

#ifndef DO_GRAPHICS

	fprintf(fpt,
		"\n\nGraph 1: Transverse abberration relative to theoretical\n");
	fprintf(fpt, "\tdiffraction disk radius");
	fprintf(fpt, "\tPlot points:");
	for (i=0; i < nzones; i++) {
		if (i % 3 == 0)
			fprintf(fpt, "\n\t\t");
		fprintf(fpt, "(%6.2f,%6.2f) ",(float) M->hm[i], (float) M->lamdafrho[i]);
	}
	fprintf(fpt,"\n\nGraph 2: Wavefront deviations from spherical wavefront\n");
	fprintf(fpt, "\tPlot points:");
	for (i=0; i <= nzones; i++) {
		if (i % 3 == 0)
			fprintf(fpt, "\n\t\t");
		fprintf(fpt, "(%6.2f,%6.2f) ", (float) M->hx[i], (float) M->wavefront[i]);
	}
#endif
	fprintf(fpt,"\n\tReference parabola: y = %g * x**2 + %g\n",
		best.a, best.c);
	fprintf(fpt, "\tpassing through (%6.2f,%6.2f) and (%6.2f,%6.2f)\n",
		M->hx[best.i], M->wavefront[best.i], M->hx[best.j], M->wavefront[best.j]);

	fprintf(fpt, "\nMaximum wavefront error = 1 / %3.1f wave at zone %d\n",
		green_light / best.deviation, best.zone);
	return;
}

void
PrFile()
{
	int ch;
	FILE *fpt;

	fpt = fopen(TXT, "r");
	while ( (ch = getc(fpt)) != EOF)
		putchar(ch);
	fclose(fpt);
	printf("Results data above is in file %s\n", TXT);
}

#define DIM(A) (sizeof(A) / sizeof(A[0]))
	/* nrand - each call returns a random variable distributed according
	 *		to the normal distribution with the mean and std deviation
	 *		given
	 */
double
nrand(double mean, double stdev)
{
	static float NormCDF[] = {
		.5000,.5040,.5080,.5120,.5160,.5199,.5239,.5279,.5319,.5359,
		.5398,.5438,.5478,.5517,.5557,.5596,.5636,.5675,.5714,.5733,
		.5793,.5832,.5871,.5910,.5948,.5987,.6026,.6064,.6103,.6141,
		.6179,.6217,.6255,.6293,.6331,.6368,.6406,.6443,.6480,.6517,
		.6554,.6591,.6628,.6664,.6700,.6736,.6772,.6808,.6844,.6879,
		.6915,.6950,.6985,.7019,.7054,.7088,.7123,.7157,.7190,.7224,
		.7257,.7291,.7324,.7357,.7389,.7422,.7454,.7486,.7517,.7549,
		.7580,.7611,.7642,.7673,.7704,.7734,.7764,.7794,.7823,.7852,
		.7881,.7910,.7939,.7967,.7995,.8023,.8051,.8078,.8106,.8133,
		.8159,.8186,.8212,.8238,.8264,.8289,.8315,.8340,.8365,.8389,
		.8413,.8438,.8461,.8485,.8508,.8531,.8554,.8577,.8599,.8621,
		.8643,.8655,.8686,.8708,.8729,.8749,.8770,.8790,.8810,.8830,
		.8849,.8869,.8888,.8907,.8925,.8944,.8962,.8980,.8997,.9015,
		.9032,.9049,.9066,.9082,.9099,.9115,.9131,.9147,.9162,.9177,
		.9192,.9207,.9222,.9236,.9251,.9265,.9279,.9292,.9306,.9319,
		.9332,.9345,.9357,.9370,.9382,.9394,.9406,.9418,.9429,.9441,
		.9452,.9463,.9474,.9484,.9495,.9505,.9515,.9525,.9535,.9545,
		.9554,.9564,.9573,.9582,.9591,.9599,.9608,.9616,.9625,.9633,
		.9641,.9649,.9656,.9664,.9671,.9678,.9686,.9693,.9699,.9706,
		.9713,.9719,.9726,.9732,.9738,.9744,.9750,.9756,.9761,.9767,
		.9772,.9778,.9783,.9788,.9793,.9798,.9803,.9808,.9812,.9817,
		.9821,.9826,.9830,.9834,.9838,.9842,.9846,.9850,.9854,.9857,
		.9861,.9864,.9868,.9871,.9875,.9878,.9881,.9884,.9887,.9890,
		.9893,.9896,.9898,.9901,.9904,.9906,.9909,.9911,.9913,.9916,
		.9918,.9920,.9922,.9925,.9927,.9929,.9931,.9932,.9934,.9936,
		.9938,.9940,.9941,.9943,.9945,.9946,.9948,.9949,.9951,.9952,
		.9953,.9955,.9956,.9957,.9959,.9960,.9961,.9962,.9963,.9964,
		.9965,.9966,.9967,.9968,.9969,.9970,.9971,.9972,.9973,.9974,
		.9974,.9975,.9976,.9977,.9977,.9978,.9979,.9979,.9980,.9981,
		.9981,.9982,.9982,.9983,.9984,.9984,.9985,.9985,.9986,.9986,
		.9987,.9987,.9987,.9988,.9988,.9989,.9989,.9989,.9990,.9990,
		.9990,.9991,.9991,.9991,.9992,.9992,.9992,.9992,.9993,.9993,
		.9993,.9993,.9994,.9994,.9994,.9994,.9994,.9995,.9995,.9995,
		.9995,.9995,.9995,.9996,.9996,.9996,.9996,.9996,.9996,.9996,
		.9997,.9997,.9997,.9997,.9997,.9997,.9997,.9997,.9997,.9998
	};
	static int first = 1;
	float uvar, nvar;
	int adjust, i;

	if (first) {
		first = 0;
		srand(1);
	}
	uvar = (float) rand() / (float) RAND_MAX;
	if (uvar < 0.5) {
		adjust = 1;
		uvar = 0.5 + uvar;
	}
	else
		adjust = 0;
	for (i = 0; i < DIM(NormCDF); i++)
		if (uvar < NormCDF[i] ) {
			nvar = i * 0.01;
			break;
		}
	if (i == DIM(NormCDF))
		nvar = DIM(NormCDF) * 0.01;	/* uvar > highest NormCDF */
	if (adjust)
		nvar = -nvar;
	return stdev * nvar + mean;
}
