                      {-- SIXTESTS.PAS, page 1 --}
{
--             +----------------------------------------+
--             |                                        |
--             |                SIXTESTS                |        PROGRAM
--             |                                        |
--             +----------------------------------------+}
{$N+}
program FIT_PARABOLA;
uses BESSEL, GRAPH, JWBLIB, TEXT_IO, MAT_OPS;
   {
   -- PURPOSE
   -- Mirror surface survey.
   
   -- NOTES
   -- Method:
   -- 1) Make spline fit to Foucault measurements.
   -- 2) Integrate dx/dy = y/(f- x) to find surface x(y),
   --    and xy^3, xy to get RHS of normal equations to estimate
   --    coefficients a, b in fit ay^2+b to x (weighted by y).
   -- 3) The weighted RMS (WRMS) is difficult to compute.  Int(yx) -
   --    (a,b)'A(a,b) seems to work.
   -- 4) The approx Strehl ratio is 1-(2WFT)^2, WFT = wavefront WRMS.
   -- 5) Complex electric field @ w=2r/L is
   --                  r
   --     T(w) = (2/r)| y exp(i(y))J0(wy/r) dy
   --                  0
   -- Energy flux density is E(w) = |T(w)|, Strehl ratio = E(0).
   -- DE for moving source is x'=(y-Y)/(X-x), surface (x,y), observation
   -- (X,Y).  For fixed source at (S,0), x' = (Df*y + Ds*(y-Y)) /
   -- (Df*(S-x) + Ds*(X-x)), Ds = (S-x) + y, Df = (X-x) + (Y-y).
   -- Get Foucault with Y0.
   -- MSWord 5.0 source code: \TP\SRC\SIXTESTS.DOC.
   
   -- MODIFICATIONS
   -- JD2448227    Jim Burrows     SURVEY_P.
   --      8228         "          merge w/PRI_FIT, APL_FOOT.
   --   2450375         "          iterate RC parms, show RMS error.
   --      0378         "          remove mirror spline.
   --      0409         "          reprogram for errors vs. parabolic.
   --      0716         "          add Strehl, integrate RHS.
   --      0740         "          integrate cos(error) for Strehl.
   --      0766         "          add diffraction computation.
   --      0772         "          add obstruction.
   --      0784         "          add profile plot.
   --      0795         "          per Nils & Mike, restrict to linear.
   --      0927         "          correct WRMS w/obstruction.
   --      0950         "          fixed/moving, Foucault/caustic.
   --      1043         "          moving source poor man's caustic.
   --      1045         "          combine all 6: (move,fix)*
   --                              (Foucault,Caustic,Poor man).
   --      1313         "          add surface output for Ronchigram.
   --      1316         "          add conic least squares fit.
   }

                      {-- SIXTESTS.PAS, page 2 --}
   type
   A2500= array[ 1.. 2500] of EXTENDED;
   VECT= array[ 1.. 8] of EXTENDED;
   VECT_50= array[ 1.. 50] of EXTENDED;
   MATRIX_F= ^A2500;
   TESTS= ( FOUCAULT, CAUSTIC, POOR_MAN);
   
   const
   DW=                   0.1;
   ENUMERATION:          array[ TESTS] of STRING=( 'Foucault',
   'caustic', 'poor-man');
   FIRST:                BOOLEAN= TRUE;
   WAVELENGTH=           550.0E-6; {-- mm.}
   
   var
   A:                    EXTENDED;
   A0, A2:               EXTENDED;
   A11, A12, A22:        EXTENDED;
   AIRY_PAST:            EXTENDED;
   B:                    EXTENDED;
   B_CONIC:              EXTENDED;
   BI_AT_Y:              VECT_50;
   BM, BN:               INTEGER;
   C:                    EXTENDED;
   CX, CY:               EXTENDED;
   COEF_X, COEF_Y:       VECT_50;
   D:                    EXTENDED;
   DX:                   EXTENDED;
   DY:                   EXTENDED;
   DBIATY:               VECT_50;
   DEVI:                 array[ 0.. 500] of EXTENDED;
   EFL:                  EXTENDED;
   CAP_X, CAP_Y, ZONE:   VECT_50;
   GGP_FILE:             TEXT;
   GRAPHDRIVER:          INTEGER;
   GRAPHMODE:            INTEGER;
   HEADING:              STRING;
   I:                    INTEGER;
   INPUT_FILE:           TEXT;
   INPUT_FILE_NAME:      STRING;
   INTEG:                EXTENDED;
   INTEG_J:              EXTENDED;
   J, K:                 INTEGER;
   KNOTS_X:              VECT_50;
   KNOTS_Y:              VECT_50;
   LEFT:                 INTEGER;
   LINES:                array[ 1.. 15] of STRING;
   LOW:                  INTEGER;
   MAG_FACT:             EXTENDED;
   MAT:                  MATRIX_F;
   MAT_2:                array[ 1.. 2, 1.. 2] of EXTENDED;
   MAT_3:                array[ 1.. 3, 1.. 3] of EXTENDED;
   MAX:                  EXTENDED;
   MAXX:                 INTEGER;
   MAXY:                 INTEGER;

                      {-- SIXTESTS.PAS, page 3 --}
   MAX_MIN:              EXTENDED;
   MIN:                  EXTENDED;
   MOVING:               BOOLEAN;
   N, N_HALF, N_KNOTS:   INTEGER;
   N_READINGS:           INTEGER;
   ORDER:                INTEGER;
   PHASE_C:              array[ 0.. 500] of EXTENDED;
   PHASE_S:              array[ 0.. 500] of EXTENDED;
   PIX_MM:               EXTENDED;
   PIX_NM:               EXTENDED;
   PIX_X_0:              INTEGER;
   PIX_X0:               INTEGER;
   PIX_X1:               INTEGER;
   PIX_Y_0:              INTEGER;
   PIX_Y0:               INTEGER;
   PIX_Y1:               INTEGER;
   PSF_PAST:             EXTENDED;
   R_C:                  EXTENDED;
   R_CONIC:              EXTENDED;
   R_OBST:               EXTENDED;
   R0, R1:               EXTENDED;
   RADIUS:               EXTENDED;
   RADIUS_P:             EXTENDED;
   RHS:                  array[ 1.. 3] of EXTENDED;
   RMS:                  EXTENDED;
   S:                    STRING;
   SIZE:                 INTEGER;
   SOURCE:               EXTENDED;
   SPREAD:               TEXT;
   SUM_C, SUM_S:         EXTENDED;
   SUM0_C, SUM0_S:       EXTENDED;
   TEST:                 TESTS;
   U, V:                 VECTOR;
   UV, W, X, XP:         EXTENDED;
   XI:                   array[ 0.. 500] of EXTENDED;
   XIN, XOUT:            VECT;
   Y:                    EXTENDED;
   Y0:                   EXTENDED;
   Y2:                   EXTENDED;
   YI:                   array[ 0.. 500] of EXTENDED;
   YP:                   EXTENDED;
   Z:                    EXTENDED;
   
   {-- Link in Borland graphics driver.}
   {$L EGAVGA}
   procedure EGAVGA; external;
   

                      {-- SIXTESTS.PAS, page 4 --}
   function FIT_AT( Z: EXTENDED; POWER: INTEGER; KNOTS, COEF: VECT_50):
   EXTENDED;
      {-- PURPOSE
      -- Return spline fit at Z for spline parms POWER, KNOTS, COEF.
      -- 1POWER3.
      }
      var
      I, J:    INTEGER;
      C:  EXTENDED;
      ZP: EXTENDED;
   begin
      ZP:= Z;
      if POWER= 2 then ZP:= SQR( Z) else if POWER= 3 then ZP:= Z* SQR(
      Z);
      INTERV( KNOTS, N_KNOTS, ZP, LEFT, I);
      BSPLVD( KNOTS, ORDER, ZP, LEFT, DBIATY, 1);
      C:= 0.0;
      for J:= 1 to ORDER do if LEFT- ORDER+ J> 0 then begin
         C:= C+ DBIATY[ J]* COEF[ LEFT- ORDER+ J];
      end;
      FIT_AT:= C;
   end;
   
   function FORMAT_FLOAT( A: EXTENDED; W, D: INTEGER): STRING;
      var
      S:  STRING;
   begin
      STR( A: W: D, S);
      FORMAT_FLOAT:= S;
   end;
   
   {$F+}
   procedure FCN( Y: EXTENDED; var XVECT, XDOT);
      {-- PURPOSE
      -- Return (in XDOT) the surface derivative dx/dy, and integrands
      -- of the RHS of the least square fit at zone radius Y and current
      -- values in XVECT.
      }
      var
      A:       EXTENDED;
      B:       EXTENDED;
      CAP_X:   EXTENDED;
      CAP_Y:   EXTENDED;
      DF:      EXTENDED;
      DS:      EXTENDED;
      R0:      EXTENDED;
      R1:      EXTENDED;
      U:       EXTENDED;
      X:       EXTENDED;
      X0:      EXTENDED;
      XP:      EXTENDED;

                      {-- SIXTESTS.PAS, page 5 --}
   begin
      case TEST of
         FOUCAULT:  begin
            CAP_X:= FIT_AT( Y, 1, KNOTS_X, COEF_X);
            CAP_Y:= 0.0;
         end;
         CAUSTIC:   begin
            CAP_X:= FIT_AT( Y, 2, KNOTS_X, COEF_X);
            CAP_Y:= FIT_AT( Y, 3, KNOTS_Y, COEF_Y);
         end;
         POOR_MAN:  begin
            CAP_X:= FIT_AT( Y, 2, KNOTS_X, COEF_X);
            XP:= VECT( XVECT)[ 8];
            VECT( XDOT)[ 1]:= XP;
         end;
      end;
      X:= VECT( XVECT)[ 1];
      A:= X* Y;
      B:= SQR( Y);
      VECT( XDOT)[ 2]:= X;
      VECT( XDOT)[ 3]:= A;
      VECT( XDOT)[ 4]:= X* B;
      VECT( XDOT)[ 5]:= A* B;
      VECT( XDOT)[ 6]:= X* SQR( B);
      VECT( XDOT)[ 7]:= SQR( X)* Y;
      if MOVING then case TEST of
         FOUCAULT, CAUSTIC: VECT( XDOT)[ 1]:=( Y- CAP_Y)/( CAP_X- X);
         POOR_MAN: VECT( XDOT)[ 8]:=( 1.0+ SQR( XP))/( CAP_X- X);
      end else case TEST of
         FOUCAULT, CAUSTIC: begin
            DF:= SQRT( SQR( CAP_X- X)+ SQR( CAP_Y- Y));
            DS:= SQRT( SQR( SOURCE- X)+ SQR( Y));
            VECT( XDOT)[ 1]:=( DF* Y+ DS*( Y- CAP_Y))/( DF*( SOURCE- X)+
            DS*( CAP_X- X));
         end;
         POOR_MAN: begin
            A:= SQRT( 1.0+ SQR( XP));
            B:= SOURCE- X;
            R0:= SQRT( SQR( B)+ SQR( Y));
            X0:=( B+ XP* Y)/ A;
            B:=( Y- B* XP)/ R0;
            U:=( B* XP- SQRT( SQR( A)- SQR( B)))/( SQR( B)- 1.0);
            R1:= U*( CAP_X- X);
            VECT( XDOT)[ 8]:= 0.5* A* SQR( A)* X0*( 1.0/ R0+ 1.0/ R1)/
            R0;
         end;
      end;
   end;
   {$F-}
   

                      {-- SIXTESTS.PAS, page 6 --}
   procedure FIT_SETUP( POWER: INTEGER; READINGS: VECT_50; var KNOTS,
   COEF: VECT_50);
      {-- PURPOSE
      -- Generate spline fit (KNOTS, COEF) linear in z^POWER, 1POWER3.
      }
      var
      I, J, K: INTEGER;
      Z:  EXTENDED;
      A:  EXTENDED;
   begin
      KNOTS[ 1]:= -1.0; KNOTS[ 2]:= -1.0;
      for I:= 3 to N_KNOTS- 2 do begin
         Z:= ZONE[ I- 1];
         if POWER= 2 then Z:= SQR( Z) else if POWER= 3 then Z:= Z* SQR(
         Z);
         KNOTS[ I]:= Z;
      end;
      Z:= RADIUS_P+ 1.0;
      if POWER= 2 then Z:= SQR( Z) else if POWER= 3 then Z:= Z* SQR( Z);
      KNOTS[ N_KNOTS- 1]:= Z; KNOTS[ N_KNOTS]:= Z;
      
      for I:= 1 to SIZE do COEF[ I]:= 0.0;
      for I:= 1 to SQR( SIZE) do MAT^[ I]:= 0.0;
      for K:= 1 to N_READINGS do begin
         A:= READINGS[ K];
         Z:= ZONE[ K];
         if POWER= 2 then Z:= SQR( Z) else if POWER= 3 then Z:= Z* SQR(
         Z);
         INTERV( KNOTS, N_KNOTS, Z, LEFT, I);
         BSPLVB( KNOTS, ORDER, 1, Z, LEFT, BI_AT_Y);
         LOW:= ORDER+ 1- LEFT;
         if LOW< 1 then LOW:= 1;
         for I:= LOW to ORDER do begin
            BM:= LEFT- ORDER+ I;
            COEF[ BM]:= COEF[ BM]+ Z* A* BI_AT_Y[ I];
            for J:= LOW to ORDER do begin
               BN:= LEFT- ORDER+ J;
               MAT^[ SIZE*( BM- 1)+ BN]:= MAT^[ SIZE*( BM- 1)+ BN]+ Z*
               BI_AT_Y[ I]* BI_AT_Y[ J];
            end;
         end;
      end;
      if MAT^[ 1]= 0.0 then begin
         MAT^[ 1]:= 1.0;
         COEF[ 1]:= READINGS[ 1];
      end;
      MATRIX_SIZE:= SIZE;
      CHOL( MAT^, I);
      if I= 0 then begin
         SOLVE( MAT^, COEF);
      end else begin
         WRITELN( ' This shouldn''t happen, please e-mail '+
         'Jim Burrows, burrjaw@halcyon.com');
         HALT;
      end;
   end;

                      {-- SIXTESTS.PAS, page 7 --}
begin
   {-- Open data file (name on command line), complain if not there.}
   if PARAMCOUNT<> 1 then begin
      WRITELN( 'Need name of data file on command line.');
      HALT;
   end;
   ASSIGN( INPUT_FILE, PARAMSTR( 1));
   {$I-}
   RESET( INPUT_FILE);
   {$I+}
   if IORESULT<> 0 then begin
      WRITELN( PARAMSTR( 1)+ ' not found.');
      HALT;
   end;
   {-- Read input data file.}
   READLN( INPUT_FILE, HEADING);
   WRITELN;
   WRITELN( HEADING);
   LINES[ 1]:= HEADING;
   READLN( INPUT_FILE, INPUT_STRING);
   GET_REAL( RADIUS_P);
   RADIUS_P:= 0.5* RADIUS_P;
   S:= '';
   GET_STRING( S);
   VAL( S, R_OBST, I);
   if I= 0 then R_OBST:= 0.5* R_OBST else if( S[ I]= #9)or( S[ I]= #32)
   then begin
      INPUT_STRING:= S;
      GET_REAL( R_OBST);
      R_OBST:= 0.5* R_OBST;
   end else R_OBST:= 0.0;
   
   {-- Get test type.}
   READLN( INPUT_FILE, S);
   case UPCASE( S[ 1]) of
      'F':     TEST:= FOUCAULT;
      'C':     TEST:= CAUSTIC;
      'P':     TEST:= POOR_MAN;
   else
      WRITELN( 'Only tests: Foucault, Caustic, "Poor man''s caustic"');
      HALT;
   end;
   READLN( INPUT_FILE, SOURCE);
   MOVING:=( SOURCE= 0.0);
   if MOVING then LINES[ 2]:= 'Moving-source ' else LINES[ 2]:=
   'Fixed-source ';
   LINES[ 2]:= LINES[ 2]+ ENUMERATION[ TEST];
   WRITELN( LINES[ 2]);
   READLN( INPUT_FILE);
   

                      {-- SIXTESTS.PAS, page 8 --}
   {-- Read and check measurements.}
   N_READINGS:= 0;
   A:= 0.0;
   while not EOF( INPUT_FILE) do begin
      READLN( INPUT_FILE, INPUT_STRING);
      GET_REAL( Z);
      GET_REAL( CX);
      if TEST= CAUSTIC then GET_REAL( CY) else CY:= 0.0;
      INC( N_READINGS);
      if Z<= A then begin
         WRITELN( 'Input error:  repeated zone radius or wrong order.');
         HALT;
      end;
      A:= Z;
      ZONE[ N_READINGS]:= Z;
      CAP_X[ N_READINGS]:= CX;
      CAP_Y[ N_READINGS]:= CY;
   end;
   CLOSE( INPUT_FILE);
   if N_READINGS< 2 then begin
      WRITELN( 'Need at least 2 measurements, more is merrier.');
      HALT;
   end;
   
   {-- Set up interpolation of readings.}
   ORDER:= 2;
   N_KNOTS:= N_READINGS+ 2;
   SIZE:= N_READINGS;
   NEW( MAT);
   case TEST of
      FOUCAULT: FIT_SETUP( 1, CAP_X, KNOTS_X, COEF_X);
      CAUSTIC: begin
         FIT_SETUP( 2, CAP_X, KNOTS_X, COEF_X);
         FIT_SETUP( 3, CAP_Y, KNOTS_Y, COEF_Y);
      end;
      POOR_MAN: FIT_SETUP( 2, CAP_X, KNOTS_X, COEF_X);
   end;
   

                      {-- SIXTESTS.PAS, page 9 --}
   {-- Integrate surface DE and RHS of normal equations.}
   Y0:= R_OBST; Y:= R_OBST; N:= 0;
   for I:= 1 to 8 do XIN[ I]:= 0.0;
   while Y<= RADIUS_P do begin
      case TEST of
         FOUCAULT, CAUSTIC: RKINIT( FCN, FIRST, Y0, Y, XIN, XOUT, 7);
         POOR_MAN: RKINIT( FCN, FIRST, Y0, Y, XIN, XOUT, 8);
      end;
      
      X:= XOUT[ 1];
      XI[ N]:= X;
      YI[ N]:= Y;
      INC( N);
      Y:= Y+ 1.0;
   end;
   if Y<> RADIUS_P+ 1 then begin
      Y:= RADIUS_P;
      case TEST of
         FOUCAULT, CAUSTIC: RKINIT( FCN, FIRST, Y0, Y, XIN, XOUT, 7);
         POOR_MAN: RKINIT( FCN, FIRST, Y0, Y, XIN, XOUT, 8);
      end;
      X:= XOUT[ 1];
      XI[ N]:= X;
      YI[ N]:= Y;
   end else DEC( N);
   
   {-- Analytic normal matrix.}
   A:= SQR( RADIUS_P);
   B:= SQR( R_OBST);
   A11:= 0.5*( A- B);
   A12:= 0.25*( SQR( A)- SQR( B));
   A22:=( A* SQR( A)- B* SQR( B))/ 6.0;
   MAT_2[ 1, 1]:= A11;
   MAT_2[ 1, 2]:= A12; MAT_2[ 2, 1]:= A12;
   MAT_2[ 2, 2]:= A22;
   MATRIX_SIZE:= 2;
   CHOL( MAT_2, I);
   RHS[ 1]:= XOUT[ 3];
   RHS[ 2]:= XOUT[ 5];
   SOLVE( MAT_2, RHS);
   

                     {-- SIXTESTS.PAS, page 10 --}
   {-- Evaluate WRMS.}
   A0:= RHS[ 1]; A2:= RHS[ 2];
   RMS:= XOUT[ 7]- A11* SQR( A0)- 2.0* A12* A0* A2- A22* SQR( A2);
   RMS:= SQRT( ABS( RMS)/ A11);  {-- 0927}
   EFL:= 0.25/ A2;
   RADIUS:= 2.0* EFL;
   
   {-- Added 1316: conic section fit x = y/2R + (b+1)y^4/8R^3 + ...}
   A11:= RADIUS_P- R_OBST;
   C:= A* RADIUS_P; D:= B* R_OBST;
   A12:=( C- D)/ 3.0;
   C:= A* C; D:= B* D;
   A22:= 0.2*( C- D);
   MAT_3[ 1, 1]:= A11;
   MAT_3[ 1, 2]:= A12; MAT_3[ 2, 1]:= A12;
   MAT_3[ 1, 3]:= A22; MAT_3[ 2, 2]:= A22; MAT_3[ 3, 1]:= A22;
   C:= A* C; D:= B* D;
   A12:=( C- D)/ 7.0; A22:=( A* C- B* D)/ 9.0;
   MAT_3[ 2, 3]:= A12; MAT_3[ 3, 2]:= A12;
   MAT_3[ 3, 3]:= A22;
   MATRIX_SIZE:= 3;
   CHOL( MAT_3, I);
   RHS[ 1]:= XOUT[ 2];
   RHS[ 2]:= XOUT[ 4];
   RHS[ 3]:= XOUT[ 6];
   SOLVE( MAT_3, RHS);
   B:= 1.0/ RHS[ 2];
   R_CONIC:= 0.5* B; B_CONIC:= RHS[ 3]* SQR( B)* B- 1.0;
   
   ASSIGN( GGP_FILE, 'MIRROR.GGP');
   REWRITE( GGP_FILE);
   WRITELN( GGP_FILE, '$ '+ HEADING);
   WRITELN( GGP_FILE, '$ ', EFL: 0: 1, ' mm FL');
   WRITELN( GGP_FILE, '+ 1 Zone radius, mm');
   WRITELN( GGP_FILE, '+ 2 CAP_X reading - R, mm');
   WRITELN( GGP_FILE, '+ 3 CAP_Y reading, mm');
   WRITELN( GGP_FILE, '+ 4 Surface x, mm');
   WRITELN( GGP_FILE, '+ 5 CAP_X fit - R, mm');
   WRITELN( GGP_FILE, '+ 6 CAP_Y fit, mm');
   WRITELN( GGP_FILE, '+ 7 Parabola CAP_X - R, mm');
   WRITELN( GGP_FILE, '+ 8 Parabola CAP_Y, mm');
   WRITELN( GGP_FILE, '+ 9 Surface error, nm');
   WRITELN( GGP_FILE, '+10 Far-field angle, "');
   WRITELN( GGP_FILE, '+11 Airy brightness, -magnitude');
   WRITELN( GGP_FILE, '+12 Pattern brightness, -magnitude');
   WRITELN( GGP_FILE, '+13 Encircled energy ratio, %');
   WRITELN( GGP_FILE, 'OBS');
   WRITELN( GGP_FILE, 'Y'#9'CAP_X'#9'CAP_Y');
   for I:= 1 to N_READINGS do WRITELN( GGP_FILE, ZONE[ I]: 0: 2, #9,
   CAP_X[ I]- RADIUS: 0: 5, #9, CAP_Y[ I]: 0: 5);
   WRITELN( GGP_FILE, '*EOF');
   

                     {-- SIXTESTS.PAS, page 11 --}
   {-- Output FONs & surface errors.}
   WRITELN( GGP_FILE, 'SURFACE');
   WRITELN( GGP_FILE, 'Y'#9, 'X'#9, 'FIT_X'#9, 'FIT_Y'#9, 'P_CAP_X',#9,
   'P_CAP_Y'#9, 'DEL_X');
   MAX:= 0.0; MIN:= 0.0;
   B:= 4.0* PI/ WAVELENGTH;
   for I:= 0 to N do begin
      Y:= YI[ I];
      X:= A0+ A2* SQR( Y);
      XP:= 2.0* A2* Y;
      D:= XI[ I]- X;
      DEVI[ I]:= D;
      PHASE_C[ I]:= Y* COS( B* D);
      PHASE_S[ I]:= Y* SIN( B* D);
      if D> MAX then MAX:= D;
      if D< MIN then MIN:= D;
      CY:= 0.0;
      case TEST of
         FOUCAULT:  CX:= FIT_AT( Y, 1, KNOTS_X, COEF_X);
         CAUSTIC: begin
            CX:= FIT_AT( Y, 2, KNOTS_X, COEF_X);
            CY:= FIT_AT( Y, 3, KNOTS_Y, COEF_Y);
         end;
         POOR_MAN: CX:= FIT_AT( Y, 2, KNOTS_X, COEF_X);
      end;

                     {-- SIXTESTS.PAS, page 12 --}
      if Y<> 0.0 then begin
         WRITE( GGP_FILE, Y: 0: 2, #9, XI[ I]: 0: 6, #9, CX- RADIUS: 0:
         5, #9, CY: 0: 5, #9);
         if MOVING then case TEST of
            FOUCAULT: WRITE( GGP_FILE, X: 0: 5, #9, 0.0: 0: 5, #9);
            CAUSTIC: WRITE( GGP_FILE, X+ SQR( Y)/ RADIUS: 0: 5, #9, -Y*
            SQR( Y)/ SQR( RADIUS): 0: 5, #9);
            POOR_MAN: WRITE( GGP_FILE, X+ SQR( Y)/ RADIUS: 0: 5, #9,
            0.0: 0: 5, #9);
         end else begin
            {-- Image point of fixed source.  This ain't easy!  Main
            -- formula is slight change from standard paraxial formula:
            -- 1/R0 + 1/R1 = (R0/X0)(2/R), with R = local radius of
            -- curvature.}
            U[ 1]:= 1.0; U[ 2]:= -XP; U[ 3]:= 0.0;
            V[ 1]:= SOURCE- X; V[ 2]:= -Y; V[ 3]:= 0.0;
            R0:= NORM( V);
            UNITIZE( U, U); UNITIZE( V, V);
            UV:= DOT( U, V);
            MUL( U, 2.0* UV);
            SUB( U, V);
            A:= 1.0+ SQR( XP);
            R_C:= RADIUS* A* SQRT( A);
            R1:= 1.0/( 2.0/ UV/ R_C- 1.0/ R0);
            case TEST of
               FOUCAULT: WRITE( GGP_FILE, X- Y* U[ 1]/ U[ 2]- RADIUS: 0:
               5, #9, 0.0: 0: 5, #9);
               CAUSTIC:  WRITE( GGP_FILE, X+ R1* U[ 1]- RADIUS: 0: 5,
               #9, Y+ R1* U[ 2]: 0: 5, #9);
               POOR_MAN: WRITE( GGP_FILE, X+ R1* U[ 1]- RADIUS: 0: 5,
               #9, 0.0: 0: 5, #9);
            end;
         end;
         WRITELN( GGP_FILE, 1.0E6* D: 0: 3);
      end;
   end;
   WRITELN( GGP_FILE, '*EOF');
   
   {-- Diffraction pattern computations.}
   MAG_FACT:= 2.5/ LN( 10.0);
   SUM0_C:= PHASE_C[ 1];
   SUM0_S:= PHASE_S[ 1];
   N_HALF:= N div 2- 1;
   

                     {-- SIXTESTS.PAS, page 13 --}
   {-- Diffraction integral @ zero angle = real Strehl ratio.  Using
   -- Simpson's rule.}
   for I:= 1 to N_HALF do begin
      J:= 2* I; K:= J+ 1;
      SUM0_C:= SUM0_C+ 4.0* PHASE_C[ J]+ 2.0* PHASE_C[ K];
      SUM0_S:= SUM0_S+ 4.0* PHASE_S[ J]+ 2.0* PHASE_S[ K];
   end;
   
   A:= 0.5*( YI[ N]- YI[ N-1]);
   SUM0_C:= SUM0_C/ 3.0 + A* PHASE_C[ N];
   SUM0_S:= SUM0_S/ 3.0 + A* PHASE_S[ N];
   if N mod 2<> 0 then begin
      SUM0_C:= SUM0_C+ PHASE_C[ N- 2]/ 6.0+( A+ 0.5)* PHASE_C[ N- 1];
      SUM0_S:= SUM0_S+ PHASE_S[ N- 2]/ 6.0+( A+ 0.5)* PHASE_S[ N- 1];
   end else begin
      SUM0_C:= SUM0_C+( A- 1.0/ 3.0)* PHASE_C[ N- 1];
      SUM0_S:= SUM0_S+( A- 1.0/ 3.0)* PHASE_S[ N- 1];
   end;
   
   WRITELN( GGP_FILE, 'FF_DIFF');
   WRITELN( GGP_FILE, 'ANG'#9'AIRY'#9'PSF'#9'EER');
   A:= SQR( 2.0/( SQR( RADIUS_P)- SQR( R_OBST)))*( SQR( SUM0_C)+ SQR(
   SUM0_S));
   WRITELN( GGP_FILE, 0.0: 0: 3, #9, 0.0: 0: 5, #9, MAG_FACT* LN( A): 0:
   5, #9, 100.0* A: 0: 5);
   INTEG:= 0.0;
   INTEG_J:= 0.0;
   
   {-- Diffraction integral at non-zero angles.  Also double integral
   -- for "EER".}
   for J:= 1 to TRUNC( 19.6/ DW) do begin
      W:= DW* J;
      B:= W/ RADIUS_P;
      C:= J0( YI[ 1]* B);
      SUM_C:= PHASE_C[ 1]* C;
      SUM_S:= PHASE_S[ 1]* C;
      

                     {-- SIXTESTS.PAS, page 14 --}
      {-- Simpson's rule.}
      for I:= 1 to N_HALF do begin
         K:= 2* I;
         C:= J0( YI[ K]* B);
         D:= J0( YI[ K+ 1]* B);
         SUM_C:= SUM_C+ 4.0* PHASE_C[ K]* C+ 2.0* PHASE_C[ K+ 1]* D;
         SUM_S:= SUM_S+ 4.0* PHASE_S[ K]* C+ 2.0* PHASE_S[ K+ 1]* D;
      end;
      A:= 0.5*( YI[ N]- YI[ N-1]);
      SUM_C:= SUM_C/ 3.0 + A* PHASE_C[ N]* J0( YI[ N]* B);
      SUM_S:= SUM_S/ 3.0 + A* PHASE_S[ N]* J0( YI[ N]* B);
      if N mod 2<> 0 then begin
         SUM_C:= SUM_C+ PHASE_C[ N- 2]* J0( YI[ N- 2]* B)/ 6.0+( A+
         0.5)* PHASE_C[ N- 1]* J0( YI[ N- 1]* B);
         SUM_S:= SUM_S+ PHASE_S[ N- 2]* J0( YI[ N- 2]* B)/ 6.0+( A+
         0.5)* PHASE_S[ N- 1]* J0( YI[ N- 1]* B);
      end else begin
         SUM_C:= SUM_C+( A- 1.0/ 3.0)* PHASE_C[ N- 1]* J0( YI[ N- 1]*
         B);
         SUM_S:= SUM_S+( A- 1.0/ 3.0)* PHASE_S[ N- 1]* J0( YI[ N- 1]*
         B);
      end;
      A:= SQR( 2.0/( SQR( RADIUS_P)- SQR( R_OBST)))*( SQR( SUM_C)+ SQR(
      SUM_S));
      if A<> 0.0 then PSF_PAST:= MAG_FACT* LN( A);
      B:= R_OBST/ RADIUS_P;
      C:= J1( W)- B* J1( B* W);
      D:= SQR( 2.0* C/( 1.0- SQR( B))/ W);
      if C<> 0.0 then AIRY_PAST:= MAG_FACT* LN( D);
      B:= 90.0* 3600.0* WAVELENGTH/ SQR( PI)/ RADIUS_P;
      if( J mod 2)<> 0 then begin
         INTEG:= INTEG+ 4.0* A* W;
         INTEG_J:= INTEG_J+ 4.0* D* W;
      end else begin
         INTEG:= INTEG+ A* W;
         INTEG_J:= INTEG_J+ D* W;
         if R_OBST= 0.0 then C:= 1.0- SQR( J0( W))- SQR( J1( W)) else
         C:= DW* INTEG_J/ 6.0;
         WRITELN( GGP_FILE, B* W: 6: 3, #9, AIRY_PAST: 0: 5, #9,
         PSF_PAST: 0: 5, #9, 100.0*( DW* INTEG/ 6.0/ C): 0: 5);
         INTEG:= INTEG+ A* W;
         INTEG_J:= INTEG_J+ D* W;
      end;
   end;
   WRITELN( GGP_FILE, '*EOF');
   CLOSE( GGP_FILE);
   

                     {-- SIXTESTS.PAS, page 15 --}
   {-- Show results.}
   LINES[ 3]:= '';
   LINES[ 4]:= 'Focal length of best fit parabola = '+ FORMAT_FLOAT(
   EFL, 0, 1)+ ' mm.';
   LINES[ 5]:= '';
   LINES[ 6]:= #9'     Surface   Wavefront';
   LINES[ 7]:= #9'          nm   (550 nm)    Goal';
   LINES[ 8]:= '';
   MAX_MIN:= MAX- MIN;
   LINES[ 9]:= #9' P-V'+ FORMAT_FLOAT( 1.0E6* MAX_MIN, 8, 1)+ #9'1/'+
   FORMAT_FLOAT( 0.5* WAVELENGTH/ MAX_MIN, 0, 1)+ #9'   < 1/4';
   LINES[ 10]:= #9' RMS'+ FORMAT_FLOAT( 1.0E6* RMS, 8, 1)+ #9'1/'+
   FORMAT_FLOAT( 0.5* WAVELENGTH/ RMS, 0, 1)+ #9'   < 1/13.4';
   LINES[ 11]:= 'Strehl ratio'#9'  -'#9+ FORMAT_FLOAT( SQR( 2.0/( SQR(
   RADIUS_P)- SQR( R_OBST)))*( SQR( SUM0_C)+ SQR( SUM0_S)), 0, 3)+
   #9'   > 0.8';
   LINES[ 12]:= '';
   LINES[ 13]:= 'Geometric conic fit:  R = '+ FORMAT_FLOAT( R_CONIC, 0,
   1)+ ' mm, deformation b = '+ FORMAT_FLOAT( B_CONIC, 0, 2);
   LINES[ 14]:= '';
   LINES[ 15]:= 'Plot file:  MIRROR.GGP.';
   for I:= 3 to 15 do WRITELN( LINES[ I]);
   WRITELN;
   WRITE( 'Hit <Enter> for mirror deviation plot> ');
   READLN;
   
   {-- Plotting code starts here.}
   MAX:= 1.0E6* MAX; MIN:= 1.0E6* MIN; MAX_MIN:= 1.0E6* MAX_MIN;
   I:= REGISTERBGIDRIVER( @EGAVGA);
   GRAPHDRIVER:= DETECT;
   INITGRAPH( GRAPHDRIVER, GRAPHMODE, '');
   MAXX:= GETMAXX;
   MAXY:= GETMAXY;
   PIX_X0:= 70;
   PIX_X1:= MAXX- 21;
   PIX_Y0:= MAXY- 47;
   PIX_Y1:= 21;
   
   {-- Draw the grid.}
   RECTANGLE( 5, 5, MAXX- 5, MAXY- 12);
   SETLINESTYLE( SOLIDLN, 0, NORMWIDTH);
   SETCOLOR( LIGHTGREEN);
   SETTEXTSTYLE( DEFAULTFONT, HORIZDIR, 1);
   

                     {-- SIXTESTS.PAS, page 16 --}
   {-- X axis grid lines, labels, and title.}
   {-- Determine "good" x-scale.}
   YI[ 0]:= YI[ 0]+ 0.001;
   YI[ N]:= YI[ N]- 0.001;
   A:= LN( YI[ N]- YI[ 0])/ LN( 10.0);
   B:= FRAC( 1.0+ FRAC( A));
   {-- 1 (.2) 1.5 (.5) 3 (1) 6 (2) 10}
   if B< 0.18 then DX:= 0.2 else if B< 0.48 then DX:= 0.5 else if B<
   0.78 then DX:= 1.0 else DX:= 2.0;
   DX:= DX* EXP( INT( A- B)* LN( 10.0));
   A:= INT(( YI[ N]- YI[ 0])/ DX);
   PIX_MM:=( PIX_X1- PIX_X0)/ DX/ A;
   if PIX_MM* FRAC( YI[ N]/ DX)* DX> 1.5 then A:= A+ 1;
   if PIX_MM* FRAC( YI[ 0]/ DX)* DX> 1.5 then A:= A+ 1;
   PIX_MM:=( PIX_X1- PIX_X0)/ DX/ A;
   A:= DX* INT( YI[ 0]/ DX);
   PIX_X_0:= ROUND( PIX_X0- A* PIX_MM);
   SETTEXTJUSTIFY( CENTERTEXT, TOPTEXT);
   SETLINESTYLE( SOLIDLN, 0, NORMWIDTH);
   LINE( PIX_X0, PIX_Y0, PIX_X0, PIX_Y1);
   STR( A: 0: 0, S);
   OUTTEXTXY( PIX_X0, PIX_Y0+ 2, S);
   A:= A+ DX;
   B:= PIX_X_0+ PIX_MM* A;
   SETLINESTYLE( DASHEDLN, 0, NORMWIDTH);
   while B< PIX_X1 do begin
      I:= ROUND( B);
      LINE( I, PIX_Y0, I, PIX_Y1);
      STR( A: 0: 0, S);
      OUTTEXTXY( I, PIX_Y0+ 2, S);
      A:= A+ DX;
      B:= PIX_X_0+ PIX_MM* A;
   end;
   SETLINESTYLE( SOLIDLN, 0, NORMWIDTH);
   LINE( PIX_X1, PIX_Y0, PIX_X1, PIX_Y1);
   STR( A: 0: 0, S);
   
   OUTTEXTXY( PIX_X1, PIX_Y0+ 2, S);
   OUTTEXTXY(( PIX_X0+ PIX_X1)div 2, PIX_Y0+ 18,
   'Zone radius, mm');
   

                     {-- SIXTESTS.PAS, page 17 --}
   {-- Y axis grid lines, labels, title.}
   {-- Determine "good" y-scale.}
   A:= LN( MAX_MIN)/ LN( 10.0);
   B:= FRAC( 1.0+ FRAC( A));
   if B< 0.18 then DY:= 0.2 else if B< 0.48 then DY:= 0.5 else if B<
   0.78 then DY:= 1.0 else DY:= 2.0;
   DY:= DY* EXP( INT( A- B)* LN( 10.0));
   A:= DY*( INT( -MIN/ DY)+ INT( MAX/ DY)+ 2.0);
   PIX_NM:=( PIX_Y0- PIX_Y1)/ A;
   
   SETTEXTJUSTIFY( RIGHTTEXT, CENTERTEXT);
   LINE( PIX_X0, PIX_Y0, PIX_X1, PIX_Y0);
   A:= DY*( INT( MIN/ DY)- 1.0);
   STR( A: 0: 0, S);
   OUTTEXTXY( PIX_X0- 1, PIX_Y0, S);
   A:= A+ DY;
   B:= PIX_Y0- PIX_NM* DY;
   SETLINESTYLE( DASHEDLN, 0, NORMWIDTH);
   while( B- PIX_Y1)> 2 do begin
      I:= ROUND( B);
      if ABS( PIX_NM* A)< 2.0 then begin
         PIX_Y_0:= I;
         SETLINESTYLE( SOLIDLN, 0, NORMWIDTH);
         SETCOLOR( LIGHTBLUE);
      end;
      LINE( PIX_X0, I, PIX_X1, I);
      STR( A: 0: 0, S);
      OUTTEXTXY( PIX_X0- 1, I, S);
      if ABS( PIX_NM* A)< 2.0 then begin
         SETCOLOR( LIGHTGREEN);
         SETLINESTYLE( DASHEDLN, 0, NORMWIDTH);
      end;
      A:= A+ DY;
      B:= B- PIX_NM* DY;
   end;
   SETLINESTYLE( SOLIDLN, 0, NORMWIDTH);
   LINE( PIX_X0, PIX_Y1, PIX_X1, PIX_Y1);
   STR( A: 0: 0, S);
   OUTTEXTXY( PIX_X0- 1, PIX_Y1, S);
   
   SETTEXTJUSTIFY( CENTERTEXT, CENTERTEXT);
   SETCOLOR( WHITE);
   OUTTEXTXY( MAXX div 2, MAXY- 5, '*** HIT <Enter> TO QUIT ***');
   
   SETCOLOR( LIGHTGREEN);
   SETTEXTSTYLE( DEFAULTFONT, VERTDIR, 1);
   SETTEXTJUSTIFY( CENTERTEXT, CENTERTEXT);
   OUTTEXTXY( 15,( PIX_Y0+ PIX_Y1)div 2,
   'Surface deviation from best-fit parabola, nm');
   

                     {-- SIXTESTS.PAS, page 18 --}
   {-- Draw the curve.}
   SETCOLOR( LIGHTRED);
   PIX_NM:= 1.0E6* PIX_NM;
   J:= PIX_X_0+ ROUND( YI[ 0]* PIX_MM);
   K:= PIX_Y_0- ROUND( DEVI[ 0]* PIX_NM);
   MOVETO( J, K);
   for I:= 1 to N do begin
      J:= PIX_X_0+ ROUND( YI[ I]* PIX_MM);
      K:= PIX_Y_0- ROUND( DEVI[ I]* PIX_NM);
      LINETO( J, K);
   end;
   
   {-- Wait to quit.}
   READLN;
   CLOSEGRAPH;
   for I:= 1 to 15 do WRITELN( LINES[ I]);
end.
